/*
 The resulting effect is a rounded rectangle with a smooth,
 fading shadow around it, giving it a 3D or floating appearance

 In this shader int data type is used instead of bool,
 because bool is allways true on android for unknown reason
 for this case 1 - true, 0 - false
 */

#ifdef GL_ES
precision mediump float;
#endif

varying vec4 v_fragmentColor;
varying vec2 v_texCoord;
uniform sampler2D u_texture;

// draw a rect with texture instead of solid color
uniform int u_drawTexture;
// texture atlas size
uniform vec2 u_mainTextureSize;
// frame origin
uniform vec2 u_textureOrigin;
// frame size
uniform vec2 u_textureSize;
// size on screen in pixels
uniform vec2 u_visibleSize;
// is frame rotated
uniform int u_textureRotated;
// smooth edges
uniform int u_antialiasingEnabled;
// rounded corners
uniform int u_frameCorners[4];


// color of the rectangle
uniform vec4 u_color;
// color of the shadow
uniform vec4 u_shadowColor;
// alpha value of closest shadow point to the rectangle
uniform float u_startShadowAlpha;
// radius of the shadow
uniform float u_shadowRadius;
// corner radius of the rectangle
uniform float u_cornerRadius;
// alpha [0-1]
uniform float u_alpha;

// The smooth edge width for antialiasing
const float edgeSmoothWidth = 1.0;




/**
 This function calculates the distance from any point to the nearest edge of a rectangle.
 It first finds the nearest point on the rectangle's edge,
 then computes the Euclidean distance between this point and the given point.
 */
float calculateDistanceToRectangle(float px, float py, float cx, float cy, float width, float height) {
    // Calculate half-dimensions of the rectangle
    float halfWidth = width / 2.0;
    float halfHeight = height / 2.0;

    // Calculate the nearest point on the rectangle to the given point
    float nearestX = max(cx - halfWidth, min(px, cx + halfWidth));
    float nearestY = max(cy - halfHeight, min(py, cy + halfHeight));

    // Calculate the distance from the point to the nearest point on the rectangle
    float dx = px - nearestX;
    float dy = py - nearestY;

    return sqrt(dx * dx + dy * dy);
}

/**
 This function calculates the shortest distance from any point to a rounded rectangle.
 It considers both the edges and the rounded corners of the rectangle to find the minimum distance
 */
float calculateShortestDistanceToRoundedRect(float px, float py, float cx, float cy, float width, float height, float r) {
    // Calculate distances to the nearest edge
    float edgeDistance1 = calculateDistanceToRectangle(px, py, cx, cy, width, height - 2.0 * r);
    float edgeDistance2 = calculateDistanceToRectangle(px, py, cx, cy, width - 2.0 * r, height);
    float edgeDistance = min(edgeDistance1, edgeDistance2);

    // Calculate distances to each corner
    float cornerDistances[4];
    vec2 corners[4];
    // bottom left
    corners[0] = vec2(cx - width / 2.0 + r, cy + height / 2.0 - r);
    // top left
    corners[1] = vec2(cx - width / 2.0 + r, cy - height / 2.0 + r);
    // top right
    corners[2] = vec2(cx + width / 2.0 - r, cy - height / 2.0 + r);
    // bottom right
    corners[3] = vec2(cx + width / 2.0 - r, cy + height / 2.0 - r);

    for (int i = 0; i < 4; i++) {
        bool cornerEnabled = clamp(float(u_frameCorners[i]), 0.0, 1.0) == 1.0;
        float distanceToCornerCircleCenter = distance(vec2(px, py), corners[i]) - r;
        float distanceToCornerRectangle = calculateDistanceToRectangle(px, py, corners[i].x, corners[i].y, 2.0 * r, 2.0 * r);
        cornerDistances[i] = cornerEnabled? distanceToCornerCircleCenter : distanceToCornerRectangle;
    }

    // The shortest distance is the minimum of edgeDistance and all cornerDistances
    float minCornerDistance = min(min(cornerDistances[0], cornerDistances[1]), min(cornerDistances[2], cornerDistances[3]));
    return min(edgeDistance, minCornerDistance);

}

/**
 This function calculates the position of the current pixel relative to the top-left corner of the frame
 If the frame is the same size as the texture, the position is the same as the texture coordinate
 If frame is rotated, the x and y coordinates are swapped
 */
vec2 getPosition() {

    // current pixel relative to the top-left corner of the texture
    vec2 positionInMainTexture = v_texCoord * u_mainTextureSize;
    // current pixel relative to the top-left corner of the frame
    vec2 positionInFrame = positionInMainTexture - u_textureOrigin;

    if (u_textureRotated == 1) {
        // if the texture is rotated, we need to swap the x and y coordinates
        positionInFrame = vec2(positionInFrame[1], positionInFrame[0]);
    }

    vec2 positionRelative = positionInFrame / u_textureSize;
    vec2 posInPixels = positionRelative * u_visibleSize;

    return posInPixels;

}

/**
 This function calculates the shadow color based on the distance from the rounded rectangle
 The alpha value of the shadow color is determined by the distance from the rectangle
 */
vec4 getShadowColor(float distanceFromRect) {
    vec4 colorShadow = u_shadowColor;
    float alpha = ((u_shadowRadius - distanceFromRect) / u_shadowRadius) * u_startShadowAlpha;
    alpha = smoothstep(0.0, edgeSmoothWidth, alpha);  // Smooth the shadow edge
    colorShadow[3] = alpha;
    return colorShadow;
}

void main() {

    // current pixel relative to the top-left corner of the frame
    vec2 positionInFrame = getPosition();

    // center of the frame
    vec2 visibleCenter = vec2(u_visibleSize[0] / 2.0, u_visibleSize[1] / 2.0);

    // distance from current pixel to the closest point on the rectangle
    float distance = calculateShortestDistanceToRoundedRect(positionInFrame[0], positionInFrame[1], visibleCenter[0], visibleCenter[1], u_visibleSize[0] - u_shadowRadius * 2.0, u_visibleSize[1] - u_shadowRadius * 2.0 , u_cornerRadius);

    vec4 colorShadow = getShadowColor(distance);
    vec4 colorRect = u_drawTexture == 1? texture2D(u_texture, v_texCoord) : u_color;

    vec4 finalColor = vec4(1.0, 1.0, 1.0, 1.0);
    
    if(u_antialiasingEnabled == 1) {
        float edgeAlpha = smoothstep(-edgeSmoothWidth, 0.0, -distance); // Smooths rectangle edge
        finalColor = mix(colorShadow, colorRect, edgeAlpha);
    } else {
        if(distance > 0.1) finalColor = colorShadow;
        else finalColor = colorRect;
    }
    
    finalColor[3] = finalColor[3] * u_alpha;
    
    gl_FragColor = finalColor;

}
